function cdx_new = update_model_price_tranches(CDX, discounts_IMM, N, start_date_num, end_date_num, ...
                                               maturities, LGD_method, how_port_loss)
% --------------------------------------------------------------------------------------------------
% Update the model implied coupon rate (up-front payment for equity) for a tranche structure between
% 'start_date' and 'end_date'.
% --------------------------------------------------------------------------------------------------
% CDX                           ... credit index structure (see 'all_steps_in_a_row.m')
% discounts_IMM                 ... structure with discount curves matching IMM dates
% N                             ... number of points for numerical integration of Fourier transform
% start_date_num                ... datenum of start date
% end_date_num                  ... datenum of end date
% maturities                    ... which maturities to update (indicator vector), default: update all
% LGD_method                    ... method for (joint) distribution of LGDs
% how_port_loss                 ... how to calculate the portfolio loss distribution
%                                   1 ... load from external file
%                                   2 ... calculate
%                                   3 ... calculate and save in external file
% --------------------------------------------------------------------------------------------------
% sample call: update_model_price_tranches(CDX_NA_IG2, discounts_IMM, 1000, datenum('05/01/2006'), datenum('05/02/2006'), [1 1])
% --------------------------------------------------------------------------------------------------

% Extract alpha_n for LGD method two (depricated)
if isfield(CDX.LGD, 'alpha_n_rel')
    alpha_n_rel = CDX.LGD.alpha_n_rel;
else
    alpha_n_rel = 0;
end

% Up-front: Calculate portfolio loss distribution conditional on number of defaults
num_firms = 125;
G_n_storage = {};
if (LGD_method == 1)
    for n=1:(num_firms+1)
        G_n_storage{n} = binopdf_fast(0:(n-1), n-1, 1/3);
    end
elseif (LGD_method == 3)    % LGD_n follows a Markov chain
    G_n_storage = G_n_LGD_Markov(num_firms, CDX.LGD.rho_n);
end

% Compute portfolio loss distribution upfront for maximum horizon needed
if (1)
    % Determine relevant date range
    start_pos = find(CDX.dates{1} >= start_date_num, 1, 'first');
    end_pos = find(CDX.dates{1} <= end_date_num, 1, 'last');
    used_range = start_pos:end_pos;
    port_loss_dates = CDX.dates{1}(used_range);
    num_dates = length(port_loss_dates);

    % Determine maximum needed horizon
    tmp = [5 7 10];
    T_max = max(tmp(logical(maturities))) + 0.25;
    used_horizons = CDX.T{1}(used_range);
    
    if (how_port_loss == 1)     % Load from external file
        load('c:/thesis/main paper/implementation/port_dist.mat', 'loss_dist', 'num_available');
    elseif (how_port_loss >=2 )  % Calculate portfolio loss distribution
        % For each date, for each coupon payment horizon, compute portfolio loss distribution
        horizon = zeros(num_dates, T_max*4 + 1);
        start_horizon = used_horizons-floor(used_horizons*4)/4;
        for j=1:num_dates
            if (start_horizon(j) > 0)
                horizon(j,2:(T_max*4 + 1)) = start_horizon(j) + (0:0.25:(T_max-0.25));
            else
                horizon(j,2:(T_max*4 + 1)) = start_horizon(j) + (0.25:0.25:(T_max));
            end
        end
        %loss_dist = port_loss_dist2(used_dates, horizon, CDX.portfolio, CDX.index_members(used_range_index_members,:), y0, CDX.AJD_common_factor, N);
        [loss_dist, num_available] = port_loss_dist_interpolate(port_loss_dates, horizon, CDX.portfolio, CDX.index_members(used_range,:), ...
                                                                CDX.y0(used_range),  CDX.AJD_common_factor, N, 7);

        % Optionally, save portfolio loss distribution in external file                                   
        if (how_port_loss == 3)
            save('c:/thesis/main paper/implementation/port_dist.mat', 'loss_dist', 'num_available');
        end
    end
end

% Update model-implied tranche prices for each maturity of index
for i=1:length(maturities)
    if (maturities(i) == 1)       
        % Match dates with the dates from the portfolio loss distribution
        [trash, port_dates_pos] = is_member_sorted_c(CDX.dates{i}, port_loss_dates);
        port_dates_pos = port_dates_pos(port_dates_pos > 0);
        used_dates = port_loss_dates(port_dates_pos);
        
        if (length(used_dates) > 0)
            % Determine relevant date range
            start_pos_horizon = find(CDX.dates{i} >= start_date_num, 1, 'first');
            end_pos_horizon = find(CDX.dates{i} <= end_date_num, 1, 'last');
            used_range = start_pos_horizon:end_pos_horizon;
            used_dates_horizon = CDX.dates{i}(used_range);
    
            % Extract relevant discount curves
            tmp = [5 7 10];
            T_max = tmp(i) + 0.25;
            [discounts, discounts_inbetween] = get_discounts(used_dates_horizon, discounts_IMM, T_max);
 
            % Determine used horizons
            [trash, cdx_liq_pos] = is_member_sorted_c(used_dates_horizon, CDX.dates{1});
            used_horizons = CDX.T{i}(used_range);

            % Solve for model implied coupon rate (up-front payment for equity)
            npv_fixed_leg = value_fixed_leg_tranches(CDX.cut_offs, used_horizons, ...
                                                     CDX.tranche_market_price_mid{i}(used_range,:), ...
                                                     CDX.tranche_market_upfront_mid{i}(used_range,:), ...
                                                     discounts, discounts_inbetween, loss_dist(port_dates_pos,:,:),...
                                                     num_available, CDX.LGD.mean, LGD_method, alpha_n_rel, G_n_storage, CDX.liq_prem_tranches(cdx_liq_pos,:));
            npv_default_leg = value_default_leg_tranches(CDX.cut_offs, used_horizons, ...
                                                         CDX.tranche_market_price_mid{i}(used_range,:), ...
                                                         CDX.tranche_market_upfront_mid{i}(used_range,:), ...
                                                         discounts, discounts_inbetween, loss_dist(port_dates_pos,:,:), ...
                                                         num_available, CDX.LGD.mean, LGD_method, alpha_n_rel, G_n_storage);
            
                                                     
            % Create fields in CDX structure (if not already exists) for model-implied prices
            if (~isfield(CDX, 'tranche_model_price'))
                CDX.tranche_model_price{i} = zeros(length(CDX.dates{i}), length(CDX.cut_offs));
                CDX.tranche_model_price{i}(:,1) = 500;
            end
            if (~isfield(CDX, 'tranche_model_upfront'))
                CDX.tranche_model_upfront{i} = zeros(length(CDX.dates{i}), length(CDX.cut_offs));
            end
            if (~isfield(CDX, 'PV01'))
                CDX.PV01{i} = zeros(length(CDX.dates{i}), length(CDX.cut_offs));
                CDX.PV01{i}(:,1) = 1;
            end
            
            % Expand size of field for model-implied prices (if necessary)
            if (length(CDX.tranche_model_upfront) < i)
                CDX.tranche_model_upfront{i} = zeros(length(CDX.dates{i}), length(CDX.cut_offs));
            end
            if (length(CDX.tranche_model_price) < i)
                CDX.tranche_model_price{i} = zeros(length(CDX.dates{i}), length(CDX.cut_offs));  
                CDX.tranche_model_price{i}(:,1) = 500;
            end
            if isempty(CDX.tranche_model_upfront{i})
                CDX.tranche_model_upfront{i} = zeros(length(CDX.dates{i}), length(CDX.cut_offs));
            end
            if (length(CDX.PV01) < i)
                CDX.PV01{i} = zeros(length(CDX.dates{i}), length(CDX.cut_offs));;
                CDX.PV01{i}(:,1) = 1;
            end
            
            % Calculate model-implied tranche prices
            CDX.tranche_model_upfront{i}(used_range,2:length(CDX.cut_offs)) = 0;
            CDX.tranche_model_upfront{i}(used_range,1) = CDX.tranche_market_upfront_mid{i}(used_range,1) + ...
                                                         (npv_default_leg(:,1) - npv_fixed_leg(:,1)) / CDX.cut_offs(1);
            CDX.tranche_model_price{i}(used_range,2:end) = npv_default_leg(:,2:end) ./ npv_fixed_leg(:,2:end) .* ...
                                                                   CDX.tranche_market_price_mid{i}(used_range,2:end);

            % Calculate risky PV01
            if (1)
                % Need to calculate fixed_leg_pv with liq_prem
                npv_fixed_leg_no_liq = value_fixed_leg_tranches(CDX.cut_offs, used_horizons, CDX.tranche_market_price_mid{i}(used_range,:), ...
                                                                CDX.tranche_market_upfront_mid{i}(used_range,:), discounts, ...
                                                                discounts_inbetween, loss_dist(port_dates_pos,:,:), num_available, ...
                                                                CDX.LGD.mean, LGD_method, alpha_n_rel, G_n_storage, zeros(num_dates, 1));
                
                CDX.PV01{i}(used_range,1) = (npv_fixed_leg_no_liq(1) - CDX.cut_offs(1)*CDX.tranche_market_upfront_mid{i}(used_range,1)) ./ ...
                                            CDX.tranche_market_price_mid{i}(used_range,1) / CDX.cut_offs(1) * 1e4;
                CDX.PV01{i}(used_range,2:end) = npv_fixed_leg_no_liq(:,2:end) ./ CDX.tranche_market_price_mid{i}(used_range,2:end) ./ ...
                                                repmat(diff(CDX.cut_offs), length(used_range), 1)* 1e4;
            end
        end
    end
end
cdx_new = CDX;